import { Injectable } from '@angular/core';
import { MyKadData, ScanResult, ICValidation } from '../models/mykad.model';

@Injectable({
  providedIn: 'root'
})
export class MyKadService {

  // Azure Computer Vision Configuration
  private readonly AZURE_ENDPOINT = 'https://mykad.cognitiveservices.azure.com/';
  private readonly AZURE_KEY = '5uoDSM0J43A7UdCapgChRThzsiJOHaUbb366BXHaPLixpHyxhwa7JQQJ99BKACqBBLyXJ3w3AAAFACOG7eAO';

  // State codes untuk Malaysia
  private stateCodes: { [key: string]: string } = {
    '01': 'Johor', '02': 'Kedah', '03': 'Kelantan', '04': 'Melaka',
    '05': 'Negeri Sembilan', '06': 'Pahang', '07': 'Pulau Pinang',
    '08': 'Perak', '09': 'Perlis', '10': 'Selangor', '11': 'Terengganu',
    '12': 'Sabah', '13': 'Sarawak', '14': 'Wilayah Persekutuan Kuala Lumpur',
    '15': 'Wilayah Persekutuan Labuan', '16': 'Wilayah Persekutuan Putrajaya',
    '21': 'Johor', '22': 'Johor', '23': 'Johor', '24': 'Johor',
    '25': 'Kedah', '26': 'Kedah', '27': 'Kedah',
    '28': 'Kelantan', '29': 'Kelantan',
    '30': 'Melaka',
    '31': 'Negeri Sembilan', '59': 'Negeri Sembilan',
    '32': 'Pahang', '33': 'Pahang',
    '34': 'Penang', '35': 'Penang',
    '36': 'Perak', '37': 'Perak', '38': 'Perak', '39': 'Perak',
    '40': 'Perlis',
    '41': 'Selangor', '42': 'Selangor', '43': 'Selangor', '44': 'Selangor',
    '45': 'Terengganu', '46': 'Terengganu',
    '47': 'Sabah', '48': 'Sabah', '49': 'Sabah',
    '50': 'Sarawak', '51': 'Sarawak', '52': 'Sarawak', '53': 'Sarawak',
    '54': 'WP Kuala Lumpur', '55': 'WP Kuala Lumpur', '56': 'WP Kuala Lumpur', '57': 'WP Kuala Lumpur',
    '58': 'WP Labuan',
    '60': 'Negeri Sembilan',
    '82': 'Tidak Diketahui'
  };

  constructor() { }

  /**
   * Scan MyKad image using Azure Computer Vision OCR
   */
  async scanMyKad(imageFile: File): Promise<ScanResult> {
    try {
      console.log('🔷 Starting Azure Computer Vision OCR...');

      // Step 1: Submit image to Azure Read API
      const operationLocation = await this.submitImageToAzure(imageFile);
      console.log('✅ Image submitted to Azure, operation:', operationLocation);

      // Step 2: Poll for OCR results
      const text = await this.getAzureOCRResults(operationLocation);
      console.log('✅ Azure OCR completed');
      console.log('📄 OCR Raw Text:', text);

      // Step 3: Parse the text using existing logic
      const parsedData = this.parseMyKadText(text);

      return {
        success: true,
        data: parsedData,
        rawText: text,
        confidence: 99 // Azure has ~99% accuracy
      };

    } catch (error: any) {
      console.error('❌ Azure OCR Error:', error);
      return {
        success: false,
        data: null,
        rawText: '',
        error: error.message || 'Azure OCR gagal. Sila cuba lagi.'
      };
    }
  }

  /**
   * Submit image to Azure Computer Vision Read API
   */
  private async submitImageToAzure(imageFile: File): Promise<string> {
    const url = `${this.AZURE_ENDPOINT}/vision/v3.2/read/analyze`;

    // Convert File to ArrayBuffer
    const arrayBuffer = await imageFile.arrayBuffer();

    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Ocp-Apim-Subscription-Key': this.AZURE_KEY,
        'Content-Type': 'application/octet-stream'
      },
      body: arrayBuffer
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error('Azure API Error:', errorText);
      throw new Error(`Azure API Error: ${response.status} - ${errorText}`);
    }

    // Get operation location from response header
    const operationLocation = response.headers.get('Operation-Location');
    if (!operationLocation) {
      throw new Error('Operation-Location header tidak dijumpai dalam response Azure');
    }

    return operationLocation;
  }

  /**
   * Poll Azure for OCR results
   */
  private async getAzureOCRResults(operationLocation: string): Promise<string> {
    let attempts = 0;
    const maxAttempts = 30; // 30 seconds max
    const delayMs = 1000; // Poll every 1 second

    while (attempts < maxAttempts) {
      await this.delay(delayMs);
      attempts++;

      console.log(`🔄 Polling Azure results (attempt ${attempts}/${maxAttempts})...`);

      const response = await fetch(operationLocation, {
        method: 'GET',
        headers: {
          'Ocp-Apim-Subscription-Key': this.AZURE_KEY
        }
      });

      if (!response.ok) {
        throw new Error(`Azure polling error: ${response.status}`);
      }

      const result = await response.json();
      const status = result.status;

      console.log(`📊 Azure status: ${status}`);

      if (status === 'succeeded') {
        // Extract text from all lines
        const text = this.extractTextFromAzureResponse(result);
        return text;
      } else if (status === 'failed') {
        throw new Error('Azure OCR failed: ' + JSON.stringify(result));
      }

      // Status is 'running' or 'notStarted', continue polling
    }

    throw new Error('Azure OCR timeout selepas 30 saat');
  }

  /**
   * Extract text from Azure Computer Vision response
   */
  private extractTextFromAzureResponse(azureResult: any): string {
    const lines: string[] = [];

    if (azureResult.analyzeResult && azureResult.analyzeResult.readResults) {
      for (const page of azureResult.analyzeResult.readResults) {
        if (page.lines) {
          for (const line of page.lines) {
            lines.push(line.text);
          }
        }
      }
    }

    return lines.join('\n');
  }

  /**
   * Delay helper for polling
   */
  private delay(ms: number): Promise<void> {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

  /**
   * Parse OCR text to extract MyKad information
   */
  private parseMyKadText(text: string): MyKadData {
    console.log('🔍 Parsing OCR text:', text);

    const lines = text.split('\n').map(l => l.trim()).filter(l => l);
    console.log('📄 Total lines:', lines.length);
    console.log('📋 Lines:', lines);

    const data: MyKadData = {
      name: '',
      icNumber: '',
      address: ''
    };

    // Extract IC Number (format: 123456-12-1234 or 123456121234)
    const icPattern = /(\d{6})[- ]?(\d{2})[- ]?(\d{4})/;
    for (const line of lines) {
      const match = line.match(icPattern);
      if (match) {
        data.icNumber = `${match[1]}-${match[2]}-${match[3]}`;
        console.log('✅ IC Number found:', data.icNumber);

        // Validate and extract info from IC
        const validation = this.validateIC(data.icNumber);
        if (validation.valid) {
          data.dateOfBirth = validation.dateOfBirth?.toLocaleDateString('ms-MY');
          data.gender = validation.gender;
          data.state = validation.stateName;
          console.log('✅ IC validated - DOB:', data.dateOfBirth, 'Gender:', data.gender, 'State:', data.state);
        }
        break;
      }
    }

    // Extract Name - Multiple strategies
    let nameFound = false;

    // Strategy 1: Look for "NAMA" or "NAME" keyword
    const nameIndex = lines.findIndex(l =>
      l.toUpperCase().includes('NAMA') || l.toUpperCase().includes('NAME')
    );

    if (nameIndex >= 0 && nameIndex + 1 < lines.length) {
      data.name = lines[nameIndex + 1].toUpperCase();
      nameFound = true;
      console.log('✅ Name found (after NAMA keyword):', data.name);
    }

    // Strategy 2: Look for line with "BIN" or "BINTI" (strong indicator)
    if (!nameFound) {
      const bintiLine = lines.find(l =>
        (l.toUpperCase().includes(' BIN ') || l.toUpperCase().includes(' BINTI ')) &&
        l.length > 10
      );
      if (bintiLine) {
        data.name = bintiLine.toUpperCase();
        nameFound = true;
        console.log('✅ Name found (BIN/BINTI):', data.name);
      }
    }

    // Strategy 3: Multi-line name (name might be split across 2 lines)
    // Example: "MOHAMAD ROSLI BIN" on one line, "ABDULLAH" on next line
    if (!nameFound) {
      for (let i = 0; i < lines.length - 1; i++) {
        const currentLine = lines[i].toUpperCase();
        const nextLine = lines[i + 1].toUpperCase();

        // Pattern 1: Line ends with BIN/BINTI, next line is last name
        if (
          /^[A-Z\s]+$/.test(currentLine) &&
          currentLine.length > 10 &&
          !currentLine.match(icPattern) &&
          (currentLine.endsWith(' BIN') || currentLine.endsWith(' BINTI')) &&
          /^[A-Z\s]+$/.test(nextLine) &&
          nextLine.length > 3 &&
          nextLine.length < 30 &&
          !nextLine.match(icPattern)
        ) {
          data.name = `${currentLine} ${nextLine}`;
          nameFound = true;
          console.log('✅ Name found (BIN/BINTI + next line):', data.name);
          break;
        }

        // Pattern 2: First line is first name, next line has BIN/BINTI
        if (
          /^[A-Z\s]+$/.test(currentLine) &&
          currentLine.length > 5 &&
          currentLine.length < 30 &&
          !currentLine.match(icPattern) &&
          (nextLine.includes('BIN') || nextLine.includes('BINTI')) &&
          !nextLine.match(icPattern)
        ) {
          data.name = `${currentLine} ${nextLine}`;
          nameFound = true;
          console.log('✅ Name found (multi-line):', data.name);
          break;
        }
      }
    }

    // Strategy 4: Look for long line with only letters (fallback)
    if (!nameFound) {
      const nameLine = lines.find(l =>
        l.length > 10 &&
        /^[A-Z\s]+$/.test(l) &&
        !l.includes('MALAYSIA') &&
        !l.includes('MYKAD') &&
        !l.includes('PENGENALAN') &&
        !l.includes('IDENTITY') &&
        !l.includes('CARD') &&
        !l.includes('WARGANEGARA')
      );
      if (nameLine) {
        data.name = nameLine;
        nameFound = true;
        console.log('✅ Name found (long letters line):', data.name);
      }
    }

    if (!nameFound) {
      console.log('⚠️ Name not found');
    }

    // Extract Address (biasanya selepas "Alamat" keyword)
    const addressIndex = lines.findIndex(l =>
      l.toUpperCase().includes('ALAMAT') || l.toUpperCase().includes('ADDRESS')
    );

    console.log('🏠 Address keyword index:', addressIndex);

    if (addressIndex >= 0) {
      // Ambil lines selepas "Alamat" sebagai address components
      const addressLines = lines.slice(addressIndex + 1, addressIndex + 8)
        .filter(l =>
          !l.match(icPattern) && // Bukan IC number
          !l.toUpperCase().includes('WARGANEGARA') &&
          !l.toUpperCase().includes('MALAYSIA') &&
          !l.toUpperCase().includes('KEWARGANEGARAAN') &&
          !l.toUpperCase().includes('CITIZENSHIP') &&
          !l.toUpperCase().includes('LELAKI') &&
          !l.toUpperCase().includes('PEREMPUAN') &&
          !l.toUpperCase().includes('ISLAM') &&
          !l.toUpperCase().includes('BUDDHA') &&
          !l.toUpperCase().includes('KRISTIAN') &&
          !l.toUpperCase().includes('HINDU') &&
          l.length > 2
        );

      console.log('📍 Address lines extracted:', addressLines);

      // Parse address components
      const addressComponents = this.parseAddressComponents(addressLines);

      console.log('🏘️ Address components parsed:', addressComponents);

      // Set individual address fields
      data.address1 = addressComponents.address1;
      data.address2 = addressComponents.address2;
      data.address3 = addressComponents.address3;
      data.postcode = addressComponents.postcode;
      data.city = addressComponents.city;
      data.stateAddress = addressComponents.state;

      // Set combined address (backward compatibility)
      data.address = [
        data.address1,
        data.address2,
        data.address3,
        data.postcode,
        data.city,
        data.stateAddress
      ].filter(Boolean).join(', ');

      console.log('✅ Full address:', data.address);
    } else {
      console.log('⚠️ No address keyword found');

      // FALLBACK: Detect address without keyword (Azure OCR might not detect "ALAMAT")
      // Address usually comes after name and before "WARGANEGARA"
      console.log('🔍 Trying fallback address detection...');

      // Find the LAST line that contains any part of the name
      let nameLastIndex = -1;
      if (data.name) {
        const nameParts = data.name.split(' ');
        for (let i = lines.length - 1; i >= 0; i--) {
          const lineUpper = lines[i].toUpperCase();
          // Check if this line contains any part of the name
          if (nameParts.some(part => part.length > 2 && lineUpper.includes(part))) {
            nameLastIndex = i;
            break;
          }
        }
      }

      const warganegaraIndex = lines.findIndex(l => l.toUpperCase().includes('WARGANEGARA'));

      console.log('📍 Name ends at index:', nameLastIndex, '(line:', lines[nameLastIndex], ')');
      console.log('📍 Warganegara at index:', warganegaraIndex);

      if (nameLastIndex >= 0 && warganegaraIndex >= 0 && warganegaraIndex > nameLastIndex + 1) {
        // Extract lines between name and warganegara
        const nameParts = data.name.split(' ');
        const addressLines = lines.slice(nameLastIndex + 1, warganegaraIndex)
          .filter(l =>
            !l.match(icPattern) &&
            l.length > 2 &&
            !l.toUpperCase().includes('MALAYSIA') &&
            !l.toUpperCase().includes('IDENTITY') &&
            !l.toUpperCase().includes('CARD') &&
            // Extra check: don't include lines that look like part of name
            !nameParts.some(part => part.length > 2 && l.toUpperCase() === part)
          );

        console.log('🏠 Fallback address lines:', addressLines);

        if (addressLines.length > 0) {
          // Parse address components
          const addressComponents = this.parseAddressComponents(addressLines);

          console.log('🏘️ Fallback address components:', addressComponents);

          // Set individual address fields
          data.address1 = addressComponents.address1;
          data.address2 = addressComponents.address2;
          data.address3 = addressComponents.address3;
          data.postcode = addressComponents.postcode;
          data.city = addressComponents.city;
          data.stateAddress = addressComponents.state;

          // Set combined address
          data.address = [
            data.address1,
            data.address2,
            data.address3,
            data.postcode,
            data.city,
            data.stateAddress
          ].filter(Boolean).join(', ');

          console.log('✅ Fallback address detected:', data.address);
        }
      } else {
        console.log('⚠️ Cannot determine address boundaries');
      }
    }

    // Extract other fields if found
    const religionLine = lines.find(l =>
      l.toUpperCase().includes('ISLAM') ||
      l.toUpperCase().includes('BUDDHA') ||
      l.toUpperCase().includes('KRISTIAN') ||
      l.toUpperCase().includes('HINDU')
    );
    if (religionLine) {
      data.religion = religionLine.trim();
    }

    return data;
  }

  /**
   * Parse address lines into structured components
   * Expected format on MyKad:
   * Line 1: Street address (LOT 2011)
   * Line 2: Street/Area name (LORONG LIMAU)
   * Line 3: Kampung/Area (KAMPUNG PERTANIAN)
   * Line 4: Postcode + City (81000 KULAI)
   * Line 5: State (JOHOR)
   */
  private parseAddressComponents(addressLines: string[]): {
    address1: string;
    address2: string;
    address3: string;
    postcode: string;
    city: string;
    state: string;
  } {
    const result = {
      address1: '',
      address2: '',
      address3: '',
      postcode: '',
      city: '',
      state: ''
    };

    if (addressLines.length === 0) {
      return result;
    }

    // Malaysian states list for detection
    const malaysianStates = [
      'JOHOR', 'KEDAH', 'KELANTAN', 'MELAKA', 'MALACCA',
      'NEGERI SEMBILAN', 'PAHANG', 'PULAU PINANG', 'PENANG',
      'PERAK', 'PERLIS', 'SELANGOR', 'TERENGGANU',
      'SABAH', 'SARAWAK',
      'WILAYAH PERSEKUTUAN', 'WP', 'KUALA LUMPUR', 'LABUAN', 'PUTRAJAYA'
    ];

    // Pattern untuk detect postcode + city (e.g., "81000 KULAI" or "KULAI 81000")
    const postcodePattern = /\b(\d{5})\b/;

    let workingLines = [...addressLines];

    // Find and extract state (usually last line)
    for (let i = workingLines.length - 1; i >= 0; i--) {
      const line = workingLines[i].toUpperCase();
      const foundState = malaysianStates.find(state => line.includes(state));
      if (foundState) {
        result.state = workingLines[i].trim();
        workingLines.splice(i, 1);
        break;
      }
    }

    // Find and extract postcode + city
    for (let i = 0; i < workingLines.length; i++) {
      const line = workingLines[i];
      const match = line.match(postcodePattern);
      if (match) {
        result.postcode = match[1];
        // City is the rest of the line without postcode
        result.city = line.replace(match[1], '').trim();
        workingLines.splice(i, 1);
        break;
      }
    }

    // Remaining lines are address1, address2, address3
    if (workingLines.length > 0) {
      result.address1 = workingLines[0].trim();
    }
    if (workingLines.length > 1) {
      result.address2 = workingLines[1].trim();
    }
    if (workingLines.length > 2) {
      result.address3 = workingLines[2].trim();
    }

    return result;
  }

  /**
   * Validate Malaysian IC Number
   */
  validateIC(ic: string): ICValidation {
    // Remove dashes
    const cleanIC = ic.replace(/-/g, '');

    // Check length
    if (cleanIC.length !== 12) {
      return { valid: false };
    }

    // Check if all numeric
    if (!/^\d{12}$/.test(cleanIC)) {
      return { valid: false };
    }

    // Extract components
    const year = cleanIC.substring(0, 2);
    const month = cleanIC.substring(2, 4);
    const day = cleanIC.substring(4, 6);
    const stateCode = cleanIC.substring(6, 8);
    const lastDigit = parseInt(cleanIC.substring(11, 12));

    // Validate date
    const currentYear = new Date().getFullYear();
    const century = parseInt(year) > (currentYear % 100) ? 1900 : 2000;
    const fullYear = century + parseInt(year);
    const monthNum = parseInt(month);
    const dayNum = parseInt(day);

    if (monthNum < 1 || monthNum > 12) {
      return { valid: false };
    }

    if (dayNum < 1 || dayNum > 31) {
      return { valid: false };
    }

    const dob = new Date(fullYear, monthNum - 1, dayNum);

    // Validate state code
    const stateName = this.stateCodes[stateCode];

    // Determine gender (odd = male, even = female)
    const gender = lastDigit % 2 === 0 ? 'Perempuan' : 'Lelaki';

    return {
      valid: true,
      formatted: `${cleanIC.substring(0, 6)}-${cleanIC.substring(6, 8)}-${cleanIC.substring(8, 12)}`,
      dateOfBirth: dob,
      gender: gender,
      stateCode: stateCode,
      stateName: stateName || 'Tidak Diketahui'
    };
  }

  /**
   * Format IC number with dashes
   */
  formatIC(ic: string): string {
    const clean = ic.replace(/-/g, '');
    if (clean.length === 12) {
      return `${clean.substring(0, 6)}-${clean.substring(6, 8)}-${clean.substring(8, 12)}`;
    }
    return ic;
  }

  /**
   * Preprocess image for better OCR accuracy
   */
  preprocessImage(file: File): Promise<File> {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();

      reader.onload = (e: any) => {
        const img = new Image();
        img.src = e.target.result;

        img.onload = () => {
          const canvas = document.createElement('canvas');
          const ctx = canvas.getContext('2d');

          if (!ctx) {
            reject(new Error('Canvas context not available'));
            return;
          }

          canvas.width = img.width;
          canvas.height = img.height;

          // Apply image enhancements for better OCR
          ctx.filter = 'contrast(150%) brightness(110%) grayscale(100%)';
          ctx.drawImage(img, 0, 0);

          canvas.toBlob((blob) => {
            if (blob) {
              const processedFile = new File([blob], file.name, { type: 'image/png' });
              resolve(processedFile);
            } else {
              reject(new Error('Image processing failed'));
            }
          }, 'image/png');
        };

        img.onerror = () => reject(new Error('Failed to load image'));
      };

      reader.onerror = () => reject(new Error('Failed to read file'));
      reader.readAsDataURL(file);
    });
  }
}
