import { Component, EventEmitter, Output, ChangeDetectorRef } from '@angular/core';
import { CommonModule } from '@angular/common';
import { FormsModule } from '@angular/forms';
import { MyKadService } from '../../services/mykad.service';
import { MyKadData, ScanResult } from '../../models/mykad.model';

@Component({
  selector: 'app-mykad-scanner',
  standalone: true,
  imports: [CommonModule, FormsModule],
  templateUrl: './mykad-scanner.component.html',
  styleUrls: ['./mykad-scanner.component.css']
})
export class MyKadScannerComponent {
  @Output() scanComplete = new EventEmitter<MyKadData>();
  @Output() scanError = new EventEmitter<string>();

  // State management
  isLoading = false;
  loadingProgress = 0;
  loadingText = 'Sedang mengimbas MyKad...';
  showResults = false;
  showCamera = false;
  uploadProgress = 0;
  isUploading = false;
  showCapturePreview = false;
  capturedImageUrl: string | null = null;
  isCapturing = false;

  // Data
  scannedData: MyKadData | null = null;
  rawOcrText = '';
  ocrConfidence = 0;
  selectedFile: File | null = null;
  previewUrl: string | null = null;

  // Validation
  icValidation: { valid: boolean; message: string } = { valid: false, message: '' };

  // Camera stream
  private videoStream: MediaStream | null = null;

  constructor(
    private myKadService: MyKadService,
    private cdr: ChangeDetectorRef
  ) {}

  /**
   * Handle file selection from input
   */
  onFileSelected(event: Event): void {
    const input = event.target as HTMLInputElement;
    if (input.files && input.files[0]) {
      this.selectedFile = input.files[0];
      this.createPreview(this.selectedFile);
    }
  }

  /**
   * Create image preview
   */
  private createPreview(file: File): void {
    console.log('📸 Creating preview for file:', file.name, file.size, 'bytes');
    const reader = new FileReader();
    reader.onload = (e: any) => {
      this.previewUrl = e.target.result;
      console.log('✅ PreviewUrl set, length:', this.previewUrl?.length);
      // Force change detection
      this.cdr.detectChanges();
    };
    reader.readAsDataURL(file);
  }

  /**
   * Open camera modal
   */
  async openCamera(): Promise<void> {
    try {
      console.log('🎬 Starting camera initialization...');
      this.showCamera = true;

      // Wait for Angular to render the modal and video element
      console.log('⏳ Waiting 200ms for DOM render...');
      await new Promise(resolve => setTimeout(resolve, 200));

      const video = document.getElementById('cameraVideo') as HTMLVideoElement;
      console.log('📹 Video element:', video);
      console.log('📹 Video element exists?', !!video);

      if (!video) {
        console.error('❌ Video element not found in DOM after 200ms');
        alert('Error: Video element tidak dijumpai. Sila tutup modal dan cuba lagi.');
        this.closeCamera();
        return;
      }

      // Set video attributes explicitly
      video.setAttribute('autoplay', '');
      video.setAttribute('playsinline', '');
      video.muted = true; // Some browsers require muted for autoplay

      console.log('📷 Requesting camera access...');
      console.log('📷 Navigator.mediaDevices available?', !!navigator.mediaDevices);

      const constraints = {
        video: {
          facingMode: 'environment',
          width: { ideal: 1280, min: 640 },
          height: { ideal: 720, min: 480 }
        }
      };
      console.log('📷 Camera constraints:', constraints);

      this.videoStream = await navigator.mediaDevices.getUserMedia(constraints);

      console.log('✅ Camera access GRANTED!');
      console.log('📹 Stream:', this.videoStream);
      console.log('📹 Video tracks:', this.videoStream.getVideoTracks());
      console.log('📹 Track settings:', this.videoStream.getVideoTracks()[0]?.getSettings());

      // Set srcObject
      video.srcObject = this.videoStream;
      console.log('✅ Video srcObject set');

      // Force play
      try {
        await video.play();
        console.log('✅ Video.play() called successfully');
      } catch (playError) {
        console.warn('⚠️ Video.play() failed, trying alternative...', playError);
        // Alternative: wait for loadedmetadata
        await new Promise<void>((resolve) => {
          video.onloadedmetadata = async () => {
            console.log('📊 Video metadata loaded');
            console.log('📊 Video dimensions:', video.videoWidth, 'x', video.videoHeight);
            try {
              await video.play();
              console.log('✅ Video playing after metadata');
              resolve();
            } catch (err) {
              console.error('❌ Play after metadata failed:', err);
              resolve();
            }
          };
          // Timeout fallback
          setTimeout(() => {
            console.warn('⏰ Metadata timeout, resolving anyway');
            resolve();
          }, 3000);
        });
      }

      console.log('✅✅✅ Camera should be visible now!');
      console.log('📹 Final check - Video dimensions:', video.videoWidth, 'x', video.videoHeight);
      console.log('📹 Final check - Video readyState:', video.readyState);
      console.log('📹 Final check - Video paused?', video.paused);

    } catch (error: any) {
      console.error('❌ Camera initialization error:', error);
      console.error('❌ Error name:', error.name);
      console.error('❌ Error message:', error.message);

      let errorMessage = 'Tidak dapat mengakses kamera. ';
      if (error.name === 'NotAllowedError') {
        errorMessage += 'Sila beri permission untuk guna kamera.';
      } else if (error.name === 'NotFoundError') {
        errorMessage += 'Kamera tidak dijumpai pada peranti ini.';
      } else if (error.name === 'NotReadableError') {
        errorMessage += 'Kamera sedang digunakan oleh aplikasi lain.';
      } else {
        errorMessage += error.message;
      }

      alert(errorMessage);
      this.closeCamera();
    }
  }

  /**
   * Capture photo from camera
   */
  capturePhoto(): void {
    // Prevent multiple clicks
    if (this.isCapturing) {
      console.log('⚠️ Already capturing, ignoring click');
      return;
    }

    console.log('📸 Capture button clicked!');
    this.isCapturing = true;

    const video = document.getElementById('cameraVideo') as HTMLVideoElement;
    const canvas = document.getElementById('cameraCanvas') as HTMLCanvasElement;
    const ctx = canvas.getContext('2d');

    console.log('Video element:', video);
    console.log('Video dimensions:', video?.videoWidth, video?.videoHeight);

    if (!video || !canvas || !ctx) {
      console.error('Missing elements:', { video: !!video, canvas: !!canvas, ctx: !!ctx });
      alert('Error: Cannot find video or canvas element');
      this.isCapturing = false;
      return;
    }

    if (video.videoWidth === 0 || video.videoHeight === 0) {
      console.error('Video stream not ready');
      alert('Sila tunggu sebentar untuk camera siap...');
      this.isCapturing = false;
      return;
    }

    canvas.width = video.videoWidth;
    canvas.height = video.videoHeight;
    ctx.drawImage(video, 0, 0);

    console.log('Image drawn to canvas');

    canvas.toBlob((blob) => {
      if (blob) {
        console.log('Blob created:', blob.size, 'bytes');
        this.selectedFile = new File([blob], 'mykad-capture.png', { type: 'image/png' });

        // Create preview URL for captured image
        this.capturedImageUrl = URL.createObjectURL(blob);
        this.showCapturePreview = true;
        this.isCapturing = false;

        // Force Angular to detect changes
        this.cdr.detectChanges();

        console.log('✅ Image captured, showing preview');
      } else {
        console.error('Failed to create blob');
        alert('Error: Gagal create image blob');
        this.isCapturing = false;
        this.cdr.detectChanges();
      }
    }, 'image/png');
  }

  /**
   * Confirm captured image and close camera
   */
  confirmCapture(): void {
    console.log('✅ Confirming capture, keeping preview visible');

    // First, create the preview from the selected file
    if (this.selectedFile) {
      this.createPreview(this.selectedFile);
    }

    // Close the camera modal (but keep previewUrl so image shows in main section)
    if (this.videoStream) {
      this.videoStream.getTracks().forEach(track => track.stop());
      this.videoStream = null;
    }

    // Close modal states
    this.showCamera = false;
    this.showCapturePreview = false;

    // Clean up captured URL (we have previewUrl now from createPreview)
    if (this.capturedImageUrl) {
      URL.revokeObjectURL(this.capturedImageUrl);
      this.capturedImageUrl = null;
    }

    console.log('✅ Preview should be visible in main section now');
  }

  /**
   * Retake photo (discard current capture)
   */
  retakePhoto(): void {
    this.showCapturePreview = false;
    this.capturedImageUrl = null;
    this.selectedFile = null;
    // Camera stays open for retake
  }

  /**
   * Close camera and stop stream
   */
  closeCamera(): void {
    if (this.videoStream) {
      this.videoStream.getTracks().forEach(track => track.stop());
      this.videoStream = null;
    }
    this.showCamera = false;
    this.showCapturePreview = false;
    this.capturedImageUrl = null;
  }

  /**
   * Start OCR scanning process
   */
  async scanNow(): Promise<void> {
    if (!this.selectedFile) {
      alert('Sila pilih gambar MyKad terlebih dahulu');
      return;
    }

    this.isUploading = true;
    this.uploadProgress = 0;
    this.isLoading = true;
    this.loadingProgress = 0;
    this.showResults = false;

    try {
      // Upload/preprocessing phase (0-30%)
      this.loadingText = 'Memproses gambar...';
      const uploadInterval = setInterval(() => {
        if (this.uploadProgress < 30) {
          this.uploadProgress += 5;
        }
      }, 150);

      const processedFile = await this.myKadService.preprocessImage(this.selectedFile);
      clearInterval(uploadInterval);
      this.uploadProgress = 40;
      await new Promise(resolve => setTimeout(resolve, 200));

      // Image processing phase (40-70%)
      const processingInterval = setInterval(() => {
        if (this.uploadProgress < 70) {
          this.uploadProgress += 5;
        }
      }, 200);

      this.uploadProgress = 100;
      clearInterval(processingInterval);
      this.isUploading = false;

      // Simulate OCR progress
      const progressInterval = setInterval(() => {
        if (this.loadingProgress < 90) {
          this.loadingProgress += 10;
        }
      }, 500);

      // Perform OCR
      this.loadingText = 'Mengimbas teks dari MyKad...';
      const result: ScanResult = await this.myKadService.scanMyKad(processedFile);

      clearInterval(progressInterval);
      this.loadingProgress = 100;

      if (result.success && result.data) {
        this.scannedData = result.data;
        this.rawOcrText = result.rawText;
        this.ocrConfidence = result.confidence || 0;

        // Validate IC number
        if (this.scannedData.icNumber) {
          const validation = this.myKadService.validateIC(this.scannedData.icNumber);
          this.icValidation = {
            valid: validation.valid,
            message: validation.valid ? '✅ IC Number sah' : '❌ IC Number tidak sah'
          };
        }

        this.showResults = true;
        this.scanComplete.emit(this.scannedData);
      } else {
        throw new Error(result.error || 'Gagal mengimbas MyKad');
      }

    } catch (error: any) {
      console.error('Scan error:', error);
      alert('Ralat: ' + error.message);
      this.scanError.emit(error.message);
    } finally {
      this.isLoading = false;
      this.loadingProgress = 0;
      this.isUploading = false;
      this.uploadProgress = 0;
    }
  }

  /**
   * Copy text to clipboard
   */
  copyToClipboard(fieldId: string): void {
    const element = document.getElementById(fieldId) as HTMLInputElement | HTMLTextAreaElement;
    if (element) {
      element.select();
      document.execCommand('copy');
      alert('Disalin ke clipboard!');
    }
  }

  /**
   * Auto-fill external form
   */
  autoFillForm(): void {
    if (this.scannedData) {
      // Emit data untuk parent component
      this.scanComplete.emit(this.scannedData);
      alert('✅ Data telah dihantar untuk auto-fill form!');
    }
  }

  /**
   * Reset scanner for new scan
   */
  resetScanner(): void {
    this.selectedFile = null;
    this.previewUrl = null;
    this.scannedData = null;
    this.rawOcrText = '';
    this.showResults = false;
    this.icValidation = { valid: false, message: '' };
  }

  /**
   * Back to home (reset everything)
   */
  backToHome(): void {
    this.resetScanner();
    // Scroll to top
    window.scrollTo({ top: 0, behavior: 'smooth' });
  }

  /**
   * Download scanned data as JSON
   */
  downloadData(): void {
    if (this.scannedData) {
      const dataStr = JSON.stringify(this.scannedData, null, 2);
      const dataBlob = new Blob([dataStr], { type: 'application/json' });
      const url = URL.createObjectURL(dataBlob);

      const link = document.createElement('a');
      link.href = url;
      link.download = `mykad-${this.scannedData.icNumber || 'data'}.json`;
      link.click();

      URL.revokeObjectURL(url);
    }
  }
}
